/*-
 * Copyright (c) 2007 Robert N. M. Watson
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE AUTHOR OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 *
 * $FreeBSD$
 */

#include <sys/types.h>
#include <sys/sysctl.h>
#include <sys/user.h>

#include <err.h>
#include <stdio.h>
#include <string.h>

#include "procstat.h"

void
procstat_basic(pid_t pid)
{
	struct kinfo_proc kip;
	int name[4];
	size_t len;

	name[0] = CTL_KERN;
	name[1] = KERN_PROC;
	name[2] = KERN_PROC_PID;
	name[3] = pid;
	len = sizeof(kip);
	if (sysctl(name, 4, &kip, &len, NULL, 0) < 0) {
		warn("sysctl: kern.proc.pid: %d", pid);
		return;
	}
	if (len != sizeof(kip)) {
		warn("kinfo_proc mismatch");
		return;
	}
	if (kip.ki_structsize != sizeof(kip)) {
		warn("kinfo_proc structure mismatch");
		return;
	}
	if (kip.ki_pid != pid) {
		warn("kinfo_proc pid mismatch");
		return;	
	}

	if (!hflag)
		printf("%5s %5s %5s %5s %5s %3s %-8s %-8s %-12s %-13s\n",
		    "PID", "PPID", "PGID", "SID", "TSID", "THR", "LOGIN",
		    "WCHAN", "COMM", "EMUL");
	printf("%5d ", kip.ki_pid);
	printf("%5d ", kip.ki_ppid);
	printf("%5d ", kip.ki_pgid);
	printf("%5d ", kip.ki_sid);
	printf("%5d ", kip.ki_tsid);
	printf("%3d ", kip.ki_numthreads);
	printf("%-8s ", kip.ki_login);
	printf("%-8s ", strlen(kip.ki_wmesg) ? kip.ki_wmesg : "-");
	printf("%-12s ", kip.ki_comm);
	printf("%-13s\n", kip.ki_emul);
}
