/*-
 * Copyright (c) 2007 Robert N. M. Watson
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE AUTHOR OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 *
 * $FreeBSD$
 */

#include <sys/types.h>
#include <sys/sysctl.h>
#include <sys/user.h>

#include <err.h>
#include <stdio.h>
#include <stdlib.h>
#include <sysexits.h>
#include <unistd.h>

#include "procstat.h"

static int aflag, bflag, cflag, fflag, kflag, sflag, tflag, vflag;
int	hflag;

static void
usage(void)
{

	fprintf(stderr, "usage: procstat [-ah] [-b | -c | -f | -k | -s | -t "
	    "| -v] [pid ...]\n");
	exit(EX_USAGE);
}

/*
 * If -a is requested, then the calling context will already have queried a
 * kinfo_proc for the process, so pass that in in case the function was going
 * to query it itself.  Otherwise, we pass NULL and it has to do the work
 * anyway.
 */
static void
procstat(pid_t pid, struct kinfo_proc *kipp)
{

	if (bflag)
		procstat_bin(pid, kipp);
	else if (cflag)
		procstat_args(pid, kipp);
	else if (fflag)
		procstat_files(pid, kipp);
	else if (kflag)
		procstat_kstack(pid, kipp, kflag);
	else if (sflag)
		procstat_cred(pid, kipp);
	else if (tflag)
		procstat_threads(pid, kipp);
	else if (vflag)
		procstat_vm(pid, kipp);
	else
		procstat_basic(pid, kipp);
}

/*
 * Sort processes first by pid and then tid.
 */
static int
kinfo_proc_compare(const void *a, const void *b)
{
	int i;

	i = ((struct kinfo_proc *)a)->ki_pid -
	    ((struct kinfo_proc *)b)->ki_pid;
	if (i != 0)
		return (i);
	i = ((struct kinfo_proc *)a)->ki_tid -
	    ((struct kinfo_proc *)b)->ki_tid;
	return (i);
}

void
kinfo_proc_sort(struct kinfo_proc *kipp, int count)
{

	qsort(kipp, count, sizeof(*kipp), kinfo_proc_compare);
}

int
main(int argc, char *argv[])
{
	struct kinfo_proc *kipp;
	int ch, i, name[4], tmp;
	size_t len;
	long l;
	pid_t pid;
	char *dummy;

	while ((ch = getopt(argc, argv, "abcfkhstv")) != -1) {
		switch (ch) {
		case 'a':
			aflag = 1;
			break;

		case 'b':
			bflag = 1;
			break;

		case 'c':
			cflag = 1;
			break;

		case 'f':
			fflag = 1;
			break;

		case 'k':
			kflag++;
			break;

		case 'h':
			hflag = 1;
			break;

		case 's':
			sflag = 1;
			break;

		case 't':
			tflag = 1;
			break;

		case 'v':
			vflag = 1;
			break;

		case '?':
		default:
			usage();
		}

	}
	argc -= optind;
	argv += optind;

	tmp = bflag + cflag + fflag + (kflag ? 1 : 0) + sflag + tflag + vflag;
	if (tmp != 0 && tmp != 1)
		usage();

	/* Must specify at least one of -a and a list of pids. */
	if (!aflag && argc < 1)
		usage();
	if (aflag) {
		name[0] = CTL_KERN;
		name[1] = KERN_PROC;
		name[2] = KERN_PROC_PROC;

		len = 0;
		if (sysctl(name, 3, NULL, &len, NULL, 0) < 0)
			err(-1, "sysctl: kern.proc.all");

		kipp = malloc(len);
		if (kipp == NULL)
			err(-1, "malloc");

		if (sysctl(name, 3, kipp, &len, NULL, 0) < 0) {
			free(kipp);
			err(-1, "sysctl: kern.proc.all");
		}
		if (len % sizeof(*kipp) != 0)
			err(-1, "kinfo_proc mismatch");
		if (kipp->ki_structsize != sizeof(*kipp))
			err(-1, "kinfo_proc structure mismatch");
		kinfo_proc_sort(kipp, len / sizeof(*kipp));
		for (i = 0; i < len / sizeof(*kipp); i++) {
			procstat(kipp[i].ki_pid, &kipp[i]);

			/* Suppress header after first process. */
			hflag = 1;
		}
		free(kipp);
	}
	for (i = 0; i < argc; i++) {
		l = strtol(argv[i], &dummy, 10);
		if (*dummy != '\0')
			usage();
		if (l < 0)
			usage();
		pid = l;

		name[0] = CTL_KERN;
		name[1] = KERN_PROC;
		name[2] = KERN_PROC_PID;
		name[3] = pid;

		len = 0;
		if (sysctl(name, 4, NULL, &len, NULL, 0) < 0)
			err(-1, "sysctl: kern.proc.pid: %d", pid);

		kipp = malloc(len);
		if (kipp == NULL)
			err(-1, "malloc");

		if (sysctl(name, 4, kipp, &len, NULL, 0) < 0) {
			free(kipp);
			err(-1, "sysctl: kern.proc.pid: %d", pid);
		}
		if (len != sizeof(*kipp))
			err(-1, "kinfo_proc mismatch");
		if (kipp->ki_structsize != sizeof(*kipp))
			errx(-1, "kinfo_proc structure mismatch");
		if (kipp->ki_pid != pid)
			errx(-1, "kinfo_proc pid mismatch");
		procstat(pid, kipp);
		free(kipp);

		/* Suppress header after first process. */
		hflag = 1;
	}
	exit(0);
}
