/*-
 * Copyright (c) 2003 Robert N. M. Watson
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE AUTHOR OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 */

#include <sys/types.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#include <machine/ioctl_meteor.h>

#include <aalib.h>

#include "tuner.h"
#include "video.h"

aa_context *context;
aa_renderparams *params;
int aa_live = 0;


ssize_t vidbuflen;
u_int32_t *vidbuf;

int width, height;

int quiet = 0;
int cleanup_on_sane_exit = 1;

void
assert(int bool, char *string)
{

	if (!bool) {
		if (aa_live)
			aa_close(context);
		fprintf(stderr, "ASSERTION failed: %s\n", string);
		exit(-1);
	}
}

__inline char
rgb_to_grey(u_int32_t rgb, u_char bright, u_char contrast)
{
	int r, g, b, val;

	r = rgb & 0x00ff0000 >> 16;
	g = rgb & 0x0000ff00 >> 8;
	b = rgb & 0x000000ff;

	/*
	 * I'm not a graphics guy, so there's probably something a bit
	 * more sophisticated I could be doing here.
	 */
	val = (r + g + b) / 3;

#if SOFTWARE_CONTRAST_AND_BRIGHTNESS
	/*
	 * Contrast adjustments occur in a signed -127 - +127 number
	 * space, so scale.
	 */
	val -= 128;

	val *= contrast;
	val /= 128;

	/*
	 * Restore normal range.
	 */
	val += 128;
	if (val > 255)
		val = 255;
	if (val < 0)
		val = 0;

	/*
	 * Perform brightness adjustment after contrast.
	 */
	val *= bright;
	val /= 128;
#endif

	if (val > 255)
		val = 255;
	if (val < 0)
		val = 0;

	return (val);
}

void
setup_video(void)
{

	context = aa_autoinit(&aa_defparams);
	assert(context != NULL, "Failed to intialize aalib");

	aa_autoinitkbd(context, 0);
	aa_hidecursor(context);
	params = aa_getrenderparams();

	aa_live = 1;
}

void
resize_video(void)
{

	if (vidbuf != NULL)
		free(vidbuf);

	width = aa_imgwidth(context);
	height = aa_imgheight(context);
	vidbuflen = width * height * sizeof(u_int32_t);

	vidbuf = (u_int32_t *) malloc(vidbuflen);
	assert(vidbuf != NULL, "malloc of vidbuf");
	bzero(vidbuf, vidbuflen);

	assert(vid_setcapturesize(width, height) == 0, "vid_setcapturesize");

	if (!quiet)
		printf("x: %d y: %d buflen: %d\n", width, height, vidbuflen);
}

int
main(int argc, char *argv[])
{
	u_char bright, contrast, shade;
	int channel, format, source;
	int framecount, framemax;
	int event, x, y;
	u_int64_t mean;
	ssize_t len;

	if (!aa_parseoptions(NULL, NULL, &argc, argv) || argc != 1) {
		fprintf(stderr, "%s", aa_help);
		return (-1);
	}

	setup_video();

	if (vid_open("/dev/bktr") == -1) {
		aa_close(context);
		return (-1);
	}

	if (tuner_open("/dev/tuner") == -1) {
		aa_close(context);
		vid_close();
		return (-1);
	}

	source = METEOR_INPUT_DEV_SVIDEO;
	format = METEOR_FMT_AUTOMODE;
	bright = 128;
	contrast = 128;
	channel = 1;


	assert(vid_setinputsource(source) == 0, "Failed to set input source");
	assert(vid_setinputformat(format) == 0, "Failed to set input format");
	assert(vid_setbrightness(bright) == 0, "Failed to set brightness");
	assert(vid_setcontrast(contrast) == 0, "Failed to set contrast");
	assert(tuner_setchannel(channel) == 0, "Failed to set channel");
	resize_video();

	framecount = 0;
	framemax = -1;

	while (1) {
		len = vid_read((char *)vidbuf, vidbuflen);
		assert(len == vidbuflen, "vid_read returned wrong len");

		mean = 0;
		for (x = 0; x < width; x++) {
			for (y = 0; y < height; y++) {
				shade = rgb_to_grey(vidbuf[x + y * width],
				    bright, contrast);
				mean += shade;
				aa_putpixel(context, x, y, shade);
			}
		}

		if (!quiet)
			printf("Frame mean: %llu\n", mean / ((u_int64_t)x *
			    (u_int64_t)y));

		aa_render(context, params, 0, 0, aa_scrwidth(context),
		    aa_scrheight(context));
		aa_flush(context);

		while ((event = aa_getevent(context, 0)) != AA_NONE) {
			switch (event) {
			case 'b':
				if (bright > 0)
					bright--;
				assert(vid_setbrightness(bright) == 0,
				    "Failed to set brightness");
				if (!quiet)
					printf("bright: %d\n", bright);
				break;

			case 'B':
				if (bright < 255)
					bright++;
				assert(vid_setbrightness(bright) == 0,
				    "Failed to set brightness");
				if (!quiet)
					printf("bright: %d\n", bright);
				break;

			case 'c':
				if (contrast > 0)
					contrast--;
				assert(vid_setcontrast(contrast) == 0,
				    "Failed to set contrast");
				if (!quiet)
					printf("contrast: %d\n", contrast);
				break;

			case 'C':
				if (contrast < 255)
					contrast++;
				assert(vid_setcontrast(contrast) == 0,
				    "Failed to set contrast");
				if (!quiet)
					printf("contrast: %d\n", contrast);
				break;

			case 'r':
				source = METEOR_INPUT_DEV_RCA;
				assert(vid_setinputsource(source) == 0,
				    "Failed to set input source");
				if (!quiet)
					printf("Source: RCA\n");
				break;

			case 's':
				source = METEOR_INPUT_DEV_SVIDEO;
				assert(vid_setinputsource(source) == 0,
				    "Failed to set input source");
				if (!quiet)
					printf("Source: SVideo\n");
				break;

			case 't':
				source = METEOR_INPUT_DEV1;
				assert(vid_setinputsource(source) == 0,
				    "Failed to set input source");
				if (!quiet)
					printf("Source: Tuner\n");
				break;

			case '+':
				channel++;
				assert(tuner_setchannel(channel) == 0,
				    "Failed to set channel");
				if (!quiet)
					printf("Channel %d\n", channel);
				break;

			case '-':
				channel--;
				assert(tuner_setchannel(channel) == 0,
				    "Failed to set channel");
				if (!quiet)
					printf("Channel %d\n", channel);
				break;

			case 'q':
				goto out;

			case AA_RESIZE:
				if (aa_resize(context))
					resize_video();
				break;
			default:
				break;
			}
		}
		framecount++;
		if (framemax != -1 && framecount >= framemax)
			break;
	}

out:
	if (cleanup_on_sane_exit) {
		aa_close(context);
		vid_close();
	}

	return (0);
}
