/*-
 * Copyright (c) 2004 Networks Associates Technology, Inc.
 * Copyright (c) 2010 Robert N. M. Watson
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE AUTHOR OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 */

#include "config/config.h"

#include <sys/param.h>
#include <sys/mount.h>
#include <sys/stat.h>
#include <sys/statvfs.h>

#include <rpc/rpc.h>

#include <err.h>
#include <errno.h>
#include <fcntl.h>
#include <stdint.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <time.h>
#include <unistd.h>

#include <readline/readline.h>
#include <readline/history.h>

#include "libnfs/libnfs.h"

#if !HAVE_STRMODE
#include "compat/strmode.h"
#endif

/*
 * Global variables for NFS client
 */
static struct nfsmount	*nfsm;		/* Current mountpoint. */
static struct nfsnode	*nfsroot;	/* Root directory. */
static struct nfsnode	*nfscwd;	/* Current working directory. */
static char		*nfscwd_path;	/* String version of nfscwd. */

static int inline
min(int x, int y)
{

	if (x < y)
		return (x);
	return (y);
}
 
static void
usage(void)
{

	fprintf(stderr, "nfsclient [-f mntfh] [-p protocol] host "
	    "[mountpoint]\n");
	exit(-1);
}

static int
max(int x, int y)
{

	return (x > y ? x : y);
}

/*
 * Break out a string command line into an argument vector of size maxargs.
 * Is destructive on passed 'command' string.  Return the number of
 * arguments.
 */
static int
command_split(char *command, char *argv[], int maxargs)
{
	char **ap;
	int i;

	i = 0;
	for (ap = argv; (*ap = strsep(&command, " \t")) != NULL;) {
		if (**ap != '\0') {
			i++;
			if (++ap >= &argv[maxargs])
				break;
		}
	}
	return (i);
}

/*
 * Perform a recursive lookup to find a node for a path.  Assumes a valid
 * reference to a parent directory, and will return a valid reference to the
 * node.
 */
static int
nfsclient_lookup(const char *path, struct nfsnode **node)
{
	struct nfsnode *curnode, *nextnode;
	char *localpath, *segment, *next;
	int error;

	localpath = strdup(path);
	if (localpath == NULL)
		return (-1);

	/*
	 * Break down the path into segments and follow the segments.  Let
	 * the server work out "." and ".." for now.  Eventually we should
	 * construct the new canonical path on the client then walk that
	 * path.  That way things like the CWD path string can be updated
	 * consistently.
	 */
	if (localpath[0] == '/')
		curnode = nfsroot;
	else
		curnode = nfscwd;
	nfs_node_ref(curnode);
	next = localpath;
	while ((segment = strsep(&next, "/")) != NULL) {
		if (strlen(segment) == 0)	/* Ignore multiple '/'s. */
			continue;
		if (nfs_lookup(nfsm, curnode, segment, &nextnode) == -1) {
			error = errno;
			nfs_node_deref(curnode);
			free(localpath);
			errno = error;
			return (-1);
		}
		nfs_node_deref(curnode);
		curnode = nextnode;
	}
	*node = curnode;
	free(localpath);
	return (0);
}

/*
 * Given a pathname, return a directory and a name segment for the last part
 * of the path, appropriate for use with operations like rmdir, mkdir, etc.
 * If requested, also return a possibly non-NULL nfsnode pointer for the
 * existing target of the final segment.  Will return a dirnode reference
 * that must be released, and a string pointer for the segment that must be
 * free'd.  If a segnode pointer is passed and returned non-NULL, that must
 * also be released.  Returns (0) on success, and (-1) on failure; not
 * finding a target nfsnode for a segment is not considered a failure.
 */
static int
nfsclient_lookup_segment(char *path, struct nfsnode **dirnodep,
    char **segmentp, struct nfsnode **segnodep)
{
	char *cp, *fname, *localpath, *origpath;
	struct nfsnode *dirnode;
	int error;

	localpath = strdup(path);
	if (localpath == NULL)
		return (-1);
	origpath = path;

	/*
	 * Break out the path into a directory and a last segment.  This
	 * could be done better.
	 */
	cp = strrchr(localpath, '/');
	if (cp != NULL) {
		*cp = '\0';
		fname = cp + 1;
		/*
		 * If the '/' we overwrote was the first character in the
		 * string, start lookup at the root.
		 */
		if (cp != localpath)
			path = localpath;
		else
			path = "/";
	} else {
		fname = path;
		path = NULL;
	}

	/*
	 * Look up the directory, if required.
	 */
	if (path != NULL) {
		if (nfsclient_lookup(path, &dirnode) == -1) {
			error = errno;
			free(localpath);
			errno = error;
			return (-1);
		}
	} else {
		dirnode = nfscwd;
		nfs_node_ref(dirnode);
	}
	if (segnodep != NULL) {
		if (nfsclient_lookup(origpath, segnodep) != 0)
			*segnodep = NULL;
	}
	*segmentp = strdup(fname);
	if (*segmentp == NULL) {
		error = errno;
		free(localpath);
		if (segnodep != NULL && *segnodep != NULL) {
			nfs_node_deref(*segnodep);
			*segnodep = NULL;
		}
		errno = error;
		return (-1);
	}
	*dirnodep = dirnode;
	return (0);
}

#define	BUFLEN_CAT	128
static void
nfsclient_cat(int argc, char *argv[])
{
	struct nfsnode *node;
	char buf[BUFLEN_CAT + 1];
	ssize_t size;
	off_t off;

	if (argc != 2) {
		fprintf(stderr, "cat file\n");
		return;
	}
	if (nfsclient_lookup(argv[1], &node) == -1) {
		perror(argv[1]);
		return;
	}
	if (nfs_access(nfsm, node, R_OK) == -1) {
		perror(argv[1]);
		nfs_node_deref(node);
		return;
	}

	off = 0;
	while ((size = nfs_read(nfsm, node, off, buf, sizeof(buf))) > 0) {
		buf[size] = '\0';
		printf("%s", buf);
		off += size;
	}
	if (size < 0)
		perror(argv[1]);
	nfs_node_deref(node);
}

/*
 * Change the NFS client current working directory to the specified path.
 * Detect relative vs. absolute path and construct a new local string version
 * of the path using that knowledge.  Note that actually it would be
 * desirable to be a bit more intelligent here so that the local path string
 * doesn't fill with '.' and '..' segments.
 */
static void
nfsclient_cd(int argc, char *argv[])
{
	struct nfsnode *newcwd;
	struct stat sb;
	char *newpath;

	if (argc != 2) {
		fprintf(stderr, "usage: cd path\n");
		return;
	}

	/*
	 * XXXRW: It would be desirable to do better at maintaining a useful
	 * cwd path string in the relative path case.
	 */
	if (argv[1][0] == '/') {
		/* Absolute path. */
		newpath = strdup(argv[1]);
		if (newpath == NULL) {
			perror("strdup");
			return;
		}
		if (nfsclient_lookup(argv[1], &newcwd) == -1) {
			perror(argv[1]);
			free(newpath);
			return;
		}
	} else {
		/* Relative path. */
		if (asprintf(&newpath, "%s/%s", nfscwd_path, argv[1]) == -1) {
			perror("asprintf");
			return;
		}
		if (nfsclient_lookup(argv[1], &newcwd) == -1) {
			perror(argv[1]);
			free(newpath);
			return;
		}
	}
	if (nfs_getattr_cached(nfsm, newcwd, &sb) < 0)
		goto err;
	if (!(S_ISDIR(sb.st_mode))) {
		errno = ENOTDIR;
		goto err;
	}
	if (nfs_access(nfsm, newcwd, X_OK) < 0)
		goto err;
	nfs_node_deref(nfscwd);
	nfscwd = newcwd;
	free(nfscwd_path);
	nfscwd_path = newpath;
	return;
err:
	perror(argv[1]);
	nfs_node_deref(newcwd);
	free(newpath);
}

/*
 * Return current file system statistics using nfs_statvfs().
 */
#define	FILESYSTEM	"Filesystem"
#define	BLOCKS		"1K-blocks"
#define	USED		"Used"
#define	AVAIL		"Avail"
#define	CAPACITY	"Capacity"
#define	MOUNTEDON	"Mounted on"
static void
nfsclient_df(int argc, char *argv[])
{
	struct statvfs statvfs;
	char temp[40];
	int blocks_len, used_len, avail_len, capacity_len;
	int capacity;
	u_int64_t blocks, used, avail;

	if (argc != 1) {
		fprintf(stderr, "Usage: df\n");
		return;
	}

	if (nfs_statvfs(nfsm, &statvfs) == -1) {
		perror("nfs_statvfs");
		return;
	}

	blocks = (statvfs.f_blocks * statvfs.f_bsize) / 1024;
	snprintf(temp, 40, "%llu", blocks);
	blocks_len = max(strlen(temp), strlen(BLOCKS));

	used = blocks - (statvfs.f_bfree * statvfs.f_bsize) / 1024;
	snprintf(temp, 40, "%llu", used);
	used_len = max(strlen(temp), strlen(USED));

	avail = (statvfs.f_bavail * statvfs.f_bsize) / 1024;
	snprintf(temp, 40, "%llu", avail);
	avail_len = max(strlen(temp), strlen(AVAIL));

	capacity = (used * 100)/ blocks;
	snprintf(temp, 40, "%d", capacity);
	capacity_len = max(strlen(temp), strlen(CAPACITY));

	printf("%*s %*s %*s  %*s\n",
	    blocks_len, BLOCKS, used_len, USED, avail_len,
	    AVAIL, capacity_len, CAPACITY);
	printf("%*llu %*llu %*llu %*d%%\n", blocks_len, blocks, used_len,
	    used, avail_len, avail, capacity_len, capacity);
}

#define	BUFLEN_GET	4096
static void
nfsclient_get(int argc, char *argv[])
{
	struct nfsnode *dirnode, *node;
	char buf[BUFLEN_GET];
	char *segment;
	ssize_t size_read, size_written;
	int fd, i;
	off_t off;

	if (argc < 2) {
		fprintf(stderr, "get [files ...]\n");
		return;
	}
	for (i = 1; i < argc; i++) {
		if (nfsclient_lookup_segment(argv[i], &dirnode, &segment,
		    &node) == -1) {
			perror(argv[i]);
			continue;
		}
		nfs_node_deref(dirnode);
		if (node == NULL) {
			errno = ENOENT;
			perror(argv[i]);
			free(segment);
			continue;
		}
		if (nfs_access(nfsm, node, R_OK) == -1) {
			perror(argv[1]);
			free(segment);
			nfs_node_deref(node);
			continue;
		}
		fd = open(segment, O_WRONLY | O_CREAT | O_EXCL, 0600);
		if (fd < 0) {
			perror(segment);
			free(segment);
			nfs_node_deref(node);
			continue;
		}
		off = 0;
		while ((size_read = nfs_read(nfsm, node, off, buf,
		    sizeof(buf))) > 0) {
			size_written = write(fd, buf, size_read);
			if (size_written < 0) {
				perror(segment);
				close(fd);
				free(segment);
				nfs_node_deref(node);
				continue;
			}
			if (size_written == 0) {
				errno = 0;
				fprintf(stderr, "%s: EOF\n", segment);
				close(fd);
				free(segment);
				nfs_node_deref(node);
				continue;
			}
			off += min(size_read, size_written);
		}
		if (size_read < 0)
			perror(argv[1]);
		close(fd);
		free(segment);
		nfs_node_deref(node);
	}
}

/*
 * Return a list of supported commands.
 */
static void
nfsclient_help(int argc, char *argv[])
{

	printf("nfsclient\n");
	printf("\n");
	printf("Commands:\n");
	printf("  cat - Print contents of file from NFS\n");
	printf("  cd, chdir - Change working directory\n");
	printf("  df - Display file system statistics\n");
	printf("  get - Get a file from NFS\n");
	printf("  help - Command list\n");
	printf("  info - NFS mount information\n");
	printf("  ls - List files in one or more directories\n");
	printf("  mkdir - Make one or more directories\n");
	printf("  put - Put a file to NFS\n");
	printf("  pwd - Print working directory\n");
	printf("  rmdir - Remove one or more directories\n");
	printf("  rm - Remove one or more files\n");
	printf("  setcred [uid] [gid] - Set NFS credential\n");
	printf("  stat - Print file information\n");
	printf("  touch - Create empty file\n");
}

/*
 * Return some basic information about the NFS mount.  Not very much right
 * now.
 */
static void
nfsclient_info(int argc, char *argv[])
{
	char *nfs_hostname, *nfs_mountpath;

	nfs_hostname = nfs_gethostname(nfsm);
	nfs_mountpath = nfs_getmountpath(nfsm);
	printf("Connected to %s:%s\n", nfs_hostname, nfs_mountpath);
	free(nfs_hostname);
	free(nfs_mountpath);
}

/*
 * List the contents of the specified directory, optionally with more detail
 * somewhat in the style of ls(1) -l.
 */
static int
nfsclient_listdir(struct nfsnode *dirnode, int longlist)
{
	struct nfsdirent nfsdirent, *nfsdirentp;
	char modebuf[12], timebuf[128];
	struct nfsdir *nfsdir;
	struct nfsnode *node;
	struct stat sb;
	int error;

	if (nfs_opendir(nfsm, dirnode, &nfsdir) == -1)
		return (-1);
	while ((error = nfs_readdir_r(nfsdir, &nfsdirent, &nfsdirentp)) == 0) {
		if (nfsdirentp == NULL)
			break;

		if (longlist) {
			if (nfs_lookup(nfsm, dirnode, nfsdirent.d_name,
			    &node) == -1) {
				perror(nfsdirent.d_name);
				continue;
			}
			if (nfs_getattr(nfsm, node, &sb) == -1) {
				perror(nfsdirent.d_name);
				nfs_node_deref(node);
				continue;
			}
			strmode(sb.st_mode, modebuf);
#if HAVE_STRUCT_STAT_ST_ATIMESPEC
			if (strftime(timebuf, 128, "%b %d %H:%M",
			    localtime(&sb.st_mtimespec.tv_sec)) == 0)
#else
			if (strftime(timebuf, 128, "%b %d %H:%M",
			    localtime(&sb.st_mtime)) == 0)
#endif
				strcpy(timebuf, "unknown");
			nfs_node_deref(node);
			printf("%s %2d %8d %8d %6ju %s %s\n", modebuf,
			    sb.st_nlink, sb.st_uid, sb.st_gid,
			    (uintmax_t)sb.st_size, timebuf,
			    nfsdirent.d_name);
		} else
			printf("%s\n", nfsdirent.d_name);
	}
	if (error == -1) {
		nfs_closedir(nfsdir);
		return (-1);
	}
	nfs_closedir(nfsdir);
	return (0);
}

/*
 * List the contents of one or more directories, using the current working
 * directory if not specified.  Optionally use a -l argument to get the long
 * listing.  Unlike ls(1), this operates only on directories for now.
 */
static void
nfsclient_ls(int argc, char *argv[])
{
	struct nfsnode *node;
	int ch, i, lflag;

	argc--;
	argv++;
#if HAVE_OPTRESET
	optreset = 1;
#endif
	optind = 0;
	lflag = 0;
	while ((ch = getopt(argc, argv, "l")) != -1) {
		switch (ch) {
		case 'l':
			lflag = 1;
			break;
		default:
			fprintf(stderr, "ls -l files ...\n");
			return;
		}
	}
	argc -= optind;
	argv += optind;

	if (argc == 0) {
		nfsclient_listdir(nfscwd, lflag);
		return;
	}

	for (i = 0; i < argc; i++) {
		if (nfsclient_lookup(argv[i], &node) == -1) {
			perror(argv[i]);
			continue;
		}
		if (nfsclient_listdir(node, lflag) == -1)
			perror(argv[i]);
		nfs_node_deref(node);
	}
}

static void
nfsclient_put(int argc, char *argv[])
{

	if (argc < 2) {
		fprintf(stderr, "put [files ...]\n");
		return;
	}
	errno = EOPNOTSUPP;
	perror("put");
}

/*
 * List the current working directory.  Subject to limitations documented in
 * the comment for nfsclient_cd().
 */
static void
nfsclient_pwd(int argc, char *argv[])
{

	if (argc != 1)
		fprintf(stderr, "usage: pwd\n");
	else
		fprintf(stdout, "%s\n", nfscwd_path);
}

/*
 * Create one or more new directories.
 */
static void
nfsclient_mkdir(int argc, char *argv[])
{
	struct nfsnode *dirnode, *node;
	char *segment;
	int i;

	if (argc < 2) {
		fprintf(stderr, "usage: mkdir path ...\n");
		return;
	}

	for (i = 1; i < argc; i++) {
		if (nfsclient_lookup_segment(argv[i], &dirnode, &segment,
		    NULL) == -1) {
			perror(argv[1]);
			continue;
		}

		if (nfs_mkdir(nfsm, dirnode, segment, 0700, &node) == -1) {
			perror(segment);
			free(segment);
			nfs_node_deref(dirnode);
			continue;
		}
		nfs_node_deref(node);
		nfs_node_deref(dirnode);
	}
}

/*
 * Remove one or more files.
 */
static void
nfsclient_rm(int argc, char *argv[])
{
	struct nfsnode *dirnode, *node;
	struct stat sb;
	char *segment;
	int i;

	if (argc < 2) {
		fprintf(stderr, "usage: rm path ...\n");
		return;
	}

	for (i = 1; i < argc; i++) {
		if (nfsclient_lookup_segment(argv[i], &dirnode, &segment,
		    &node) == -1) {
			perror(argv[1]);
			continue;
		}
		if (node == NULL) {
			errno = ENOENT;
			perror(argv[1]);
			free(segment);
			continue;
		}
		if (nfs_getattr_cached(nfsm, node, &sb) < 0) {
			perror(argv[1]);
			free(segment);
			nfs_node_deref(dirnode);
			nfs_node_deref(node);
			continue;
		}
		nfs_node_deref(node);
		if (S_ISDIR(sb.st_mode)) {
			errno = EISDIR;
			perror(argv[1]);
			free(segment);
			nfs_node_deref(dirnode);
			continue;
		}
		nfs_node_deref(node);
		if (nfs_unlink(nfsm, dirnode, segment) == -1) {
			perror(segment);
			free(segment);
			nfs_node_deref(dirnode);
			continue;
		}
		free(segment);
		nfs_node_deref(dirnode);
	}
}

/*
 * Remove one or more directories.
 */
static void
nfsclient_rmdir(int argc, char *argv[])
{
	struct nfsnode *dirnode, *node;
	struct stat sb;
	char *segment;
	int i;

	if (argc < 2) {
		fprintf(stderr, "usage: rmdir path ...\n");
		return;
	}

	for (i = 1; i < argc; i++) {
		if (nfsclient_lookup_segment(argv[i], &dirnode, &segment,
		    &node) == -1) {
			perror(argv[1]);
			continue;
		}
		if (node == NULL) {
			errno = ENOENT;
			perror(argv[1]);
			free(segment);
			continue;
		}
		if (nfs_getattr_cached(nfsm, node, &sb) < 0) {
			perror(argv[1]);
			free(segment);
			nfs_node_deref(dirnode);
			nfs_node_deref(dirnode);
			continue;
		}
		nfs_node_deref(node);
		if (!S_ISDIR(sb.st_mode)) {
			errno = ENOTDIR;
			perror(argv[1]);
			free(segment);
			nfs_node_deref(dirnode);
			continue;
		}
		if (nfs_rmdir(nfsm, dirnode, segment) == -1) {
			perror(segment);
			free(segment);
			nfs_node_deref(dirnode);
			continue;
		}
		free(segment);
		nfs_node_deref(dirnode);
	}
}

/*
 * Modify the credential used for NFS access.
 */
static void
nfsclient_setcred(int argc, char *argv[])
{
	struct nfscred nc;
	long long ll;
	char *endp;
	uint32_t uid, gid;

	if (argc != 3) {
		fprintf(stderr, "usage: setcred uid gid\n");
		return;
	}

	ll = strtoll(argv[1], &endp, 10);
	if (ll < 0 || ll > UINT32_MAX || *endp != '\0') {
		fprintf(stderr, "usage: setcred uid gid\n");
		return;
	}
	uid = ll;

	ll = strtoll(argv[2], &endp, 10);
	if (ll < 0 || ll > UINT32_MAX || *endp != '\0') {
		fprintf(stderr, "usage: setcred uid gid\n");
		return;
	}
	gid = ll;

	bzero(&nc, sizeof(nc));
	nc.nc_uid = uid;
	nc.nc_gid = gid;

	if (nfs_setcred(nfsm, &nc) < 0) {
		perror("nfs_cred");
		return;
	}

	printf("Credential set: uid %u gid %u\n", uid, gid);
}

/*
 * Display some detailed information about the object pointed to by the
 * specified path.
 */
static void
nfsclient_stat(int argc, char *argv[])
{
	struct nfsnode *node;
	char modebuf[12];
	struct stat sb;

	if (argc != 2) {
		fprintf(stderr, "usage: stat path\n");
		return;
	}

	if (nfsclient_lookup(argv[1], &node) == -1) {
		perror(argv[1]);
		return;
	}
	if (nfs_getattr(nfsm, node, &sb) == -1) {
		perror(argv[1]);
		nfs_node_deref(node);
		return;
	}
	printf("Path: %s\n", argv[1]);
	strmode(sb.st_mode, modebuf);
	printf("Mode: %s\n", modebuf);
	printf("Links: %d\n", sb.st_nlink);
	printf("Uid: %d\n", sb.st_uid);
	printf("Gid: %d\n", sb.st_gid);
	printf("Size: %ju\n", (uintmax_t)sb.st_size);
	nfs_node_deref(node);
}

/*
 * Create an empty file.  No-op if file already exists.  Not quite touch(1)
 * as it doesn't update the time stamp.
 */
static void
nfsclient_touch(int argc, char *argv[])
{
	struct nfsnode *dirnode, *node;
	char *segment;
	int i;

	if (argc < 2) {
		fprintf(stderr, "usage: touch path ...\n");
		return;
	}

	for (i = 1; i < argc; i++) {
		if (nfsclient_lookup_segment(argv[i], &dirnode, &segment,
		    &node) == -1) {
			perror(argv[1]);
			continue;
		}
		if (node == NULL) {
			if (nfs_create(nfsm, dirnode, segment, 0600,
			    &node) == -1)
				perror(segment);
			else
				nfs_node_deref(node);
		} else {
			if (nfs_utimes(nfsm, node, NULL) == -1)
				perror(argv[1]);
			else
				nfs_node_deref(node);
		}
		free(segment);
		nfs_node_deref(dirnode);
	}
}

/*
 * Command demux.  If we get too many more, we should move to an array of
 * command names and function pointers.  Note that "exit" and "quit" are
 * actually processed in the command loop in main().
 */
static void
nfsclient_run(int argc, char *argv[])
{

	if (argc < 1)
		return;

	if (strcmp(argv[0], "cat") == 0)
		nfsclient_cat(argc, argv);
	else if (strcmp(argv[0], "cd") == 0 ||
	    strcmp(argv[0], "chdir") == 0)
		nfsclient_cd(argc, argv);
	else if (strcmp(argv[0], "df") == 0)
		nfsclient_df(argc, argv);
	else if (strcmp(argv[0], "get") == 0)
		nfsclient_get(argc, argv);
	else if (strcmp(argv[0], "help") == 0)
		nfsclient_help(argc, argv);
	else if (strcmp(argv[0], "info") == 0)
		nfsclient_info(argc, argv);
	else if (strcmp(argv[0], "ls") == 0)
		nfsclient_ls(argc, argv);
	else if (strcmp(argv[0], "mkdir") == 0)
		nfsclient_mkdir(argc, argv);
	else if (strcmp(argv[0], "put") == 0)
		nfsclient_put(argc, argv);
	else if (strcmp(argv[0], "pwd") == 0)
		nfsclient_pwd(argc, argv);
	else if (strcmp(argv[0], "rm") == 0)
		nfsclient_rm(argc, argv);
	else if (strcmp(argv[0], "rmdir") == 0)
		nfsclient_rmdir(argc, argv);
	else if (strcmp(argv[0], "setcred") == 0)
		nfsclient_setcred(argc, argv);
	else if (strcmp(argv[0], "stat") == 0)
		nfsclient_stat(argc, argv);
	else if (strcmp(argv[0], "touch") == 0)
		nfsclient_touch(argc, argv);
	else
		printf("Unknown command \"%s\"\n", argv[0]);
}

#define	MAXARGS	32
int
main(int argc, char *argv[])
{
	char *comargv[MAXARGS];
	struct nfscred nc;
	char *command, *commandline;
	char *server, *mountpoint;
	int comargc;
	char *fhandle;
	const char *nettype;
	int fflag, ch;

	fflag = 0;
	fhandle = NULL;
	nettype = "tcp";
	while ((ch = getopt(argc, argv, "f:p:")) != -1) {
		switch (ch) {
		case 'f':
			fflag++;
			fhandle = optarg;
			break;
		case 'p':
			if (strcmp(optarg, "tcp") == 0 ||
			    strcmp(optarg, "udp") == 0)
				nettype = optarg;
			else
				usage();
		case '?':
		default:
			usage();
		}
	}
	argc -= optind;
	argv += optind;

	if (fflag == 0) {
		if (argc != 2)
			usage();
	} else if (fflag == 1) {
		if (argc != 1)
			usage();
	} else
		usage();

	server = argv[0];
	if (!fflag)
		mountpoint = argv[1];
	else
		mountpoint = "";

	nfsm = NULL;
	bzero(&nc, sizeof(nc));			/* Root credential. */

	if (fflag) {
		if (nfs_mount(&nfsm, &nc, server, NFSMOUNT_TYPE_HANDLE,
		    fhandle, "tcp") == -1)
			err(-1, "nfs_mount_fh(%s, %s)", server, fhandle);
	} else {
		if (nfs_mount(&nfsm, &nc, server, NFSMOUNT_TYPE_PATH,
		    mountpoint, "tcp") == -1)
			err(-1, "nfs_mount(%s, %s)", server, mountpoint);
	}

	if (nfs_getroot(nfsm, &nfsroot) == -1) {
		perror("nfs_getroot");
		goto unmount;
	}
	nfs_node_ref(nfsroot);
	nfscwd = nfsroot;
	nfscwd_path = strdup("/");

	if (asprintf(&commandline, "%s:%s> ", server, mountpoint) == -1)
		err(-1, "asprintf");
	while ((command = readline(commandline)) != NULL) {
		if (strcmp(command, "quit") == 0 ||
		    strcmp(command, "exit") == 0) {
			free(command);
			break;
		}

		comargc = command_split(command, comargv, MAXARGS);
		if (comargc == MAXARGS)
			printf("Too many arguments\n");
		else
			nfsclient_run(comargc, comargv);
		free(command);
	}

unmount:
	if (nfscwd != NULL) {
		nfs_node_deref(nfscwd);
		nfscwd = NULL;
	}
	if (nfsroot != NULL) {
		nfs_node_deref(nfsroot);
		nfsroot = NULL;
	}
	nfs_unmount(nfsm);
	nfsm = NULL;
	return (0);
}
