#include <sys/param.h>
#include <sys/linker.h>
#include <sys/module.h>

#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#include "modulelist.moc"

#define	BUFLEN	256

struct module_entry {
	/* From the file */
	char		mef_name[64];
	char		mef_description[64];
	char		mef_module[64];
	int		mef_unload;
	int		mef_load;

	/* Run-time */
	int		mer_loaded;
	struct module_entry	*mer_next;
};

static module_entry *list, *last;

static void
ml_load_configuration(const char *filename)
{
	FILE *file;
	struct module_entry *temp, fillout;
	char buffer[BUFLEN], *cur, *tok;
	int fileid, linecount;

	file = fopen(filename, "r");
	if (file == NULL)
		return;

	/* Clear the current list, if any. */
	temp = list;
	while (temp != NULL) {
		list = temp;
		temp = temp->mer_next;
		free(list);
	}
	list = last = NULL;

	linecount = 0;
	temp = NULL;
	while (fgets(buffer, BUFLEN, file) != NULL) {
		linecount++;
		cur = buffer;

		/* Trim leading whitespace. */
		while (*cur == ' ' || *cur == '\t')
			cur++;

		/* No CR, thanks */
		if (cur[strlen(cur)-1] == '\n')
			cur[strlen(cur)-1] = '\0';

		/* Is it a comment line? */
		if (cur[0] == '#')
			continue;

		bzero(&fillout, sizeof(fillout));

		/* Module file */
		tok = strsep(&cur, ":");
		if (tok == NULL) {
			printf("%s:%d: bad module filename\n", filename,
			    linecount);
			continue;
		}
		strlcpy(fillout.mef_module, tok, sizeof(fillout.mef_module));

		/* Module name */
		tok = strsep(&cur, ":");
		if (tok == NULL) {
			printf("%s:%d: bad module name\n", filename,
			    linecount);
			continue;
		}
		strlcpy(fillout.mef_name, tok, sizeof(fillout.mef_name));

		/* Module description */
		tok = strsep(&cur, ":");
		if (tok == NULL) {
			printf("%s:%d: bad module description\n", filename,
			    linecount);
			continue;
		}
		strlcpy(fillout.mef_description, tok,
		    sizeof(fillout.mef_description));

		/* Options */
		fillout.mef_load = 1;
		fillout.mef_unload = 1;
		tok = strsep(&cur, ":");
		if (tok != NULL) {
			char *tok2;
			while((tok2 = strsep(&tok, ",")) != NULL) {
				if (strcmp(tok2, "noload") == 0)
					fillout.mef_load = 0;
				else if (strcmp(tok2, "nounload") == 0)
					fillout.mef_unload = 0;
				else if (tok2 != NULL && strlen(tok2) != 0)
					printf("%s:%d: bad option '%s'\n",
					    filename, linecount, tok2);
			}
		}

		temp = (struct module_entry *) malloc(sizeof(*temp));
		*temp = fillout;
		temp->mer_next = NULL;

		if (last) {
			last->mer_next = temp;
			last = temp;
		} else
			list = last = temp;

		fileid = kldfind(temp->mef_module);
		temp->mer_loaded = (fileid != -1);
	}

	fclose(file);
};

static void
ml_render(QListView *listview)
{
	struct module_entry *temp;

	listview->clear();

	for (temp = list; temp != NULL; temp = temp->mer_next) {
		const char *loadable, *unloadable, *loaded;

		loadable = temp->mef_load ? "yes" : "no";
		unloadable = temp->mef_unload ? "yes" : "no";
		loaded = temp->mer_loaded ? "yes" : "no";

		new QListViewItem(listview, temp->mef_module,
		    temp->mef_name, temp->mef_description, loadable,
		    unloadable, loaded);
	}
}

ModuleList::ModuleList(QWidget *parent) : QVBox(parent)
{
	QHBox *hbox;

	listview = new QListView(this, "");
	listview->addColumn("Module");
	listview->addColumn("Name");
	listview->addColumn("Description");
	listview->addColumn("Loadable");
	listview->addColumn("Unloadable");
	listview->addColumn("Loaded");

	hbox = new QHBox(this);
	loadbutton = new QPushButton("Load Module", hbox);
	unloadbutton = new QPushButton("Unload Module", hbox);

	connect(loadbutton, SIGNAL(clicked()), this,
	    SLOT(slotLoadModule()));
	connect(unloadbutton, SIGNAL(clicked()), this,
	    SLOT(slotUnloadModule()));

	refresh();
}

void
ModuleList::refresh()
{

	ml_load_configuration("modules.conf");
	ml_render(listview);
}

void
ModuleList::slotLoadModule()
{
	QListViewItem *item;

	item = listview->selectedItem();

	printf("Load! %s\n", item->text(0).latin1());

	if (kldload(item->text(0).latin1()) == -1) {
		perror("kldload");
		return;
	}

	refresh();
}

void
ModuleList::slotUnloadModule()
{
	QListViewItem *item;
	int modid;

	item = listview->selectedItem();

	printf("Unload! %s\n", item->text(0).latin1());

	modid = kldfind(item->text(0).latin1());
	if (modid == -1) {
		perror("kldfind");
		return;
	}

	if (kldunload(modid) == -1) {
		perror("kldunload");
		return;
	}

	refresh();
}
